<?php
// /api/l2/top_products.php
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

function resolve_range_or_fail(): array {
    $month_from = trim((string)get_param('month_from', ''));
    $month_to   = trim((string)get_param('month_to', ''));

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if ($month_from !== '' && $month_to !== '') {
        if (!preg_match('/^\d{4}\-\d{2}$/', $month_from) || !preg_match('/^\d{4}\-\d{2}$/', $month_to)) {
            fail('Invalid month_from/month_to. Use YYYY-MM', 400);
        }
        $date_from = $month_from . '-01';
        $date_to   = date('Y-m-t', strtotime($month_to . '-01'));
    }

    if (!$date_from || !$date_to) {
        fail('month_from/month_to (or date_from/date_to) are required', 400);
    }

    $date_to_excl = add_days($date_to, 1);
    return [$date_from, $date_to, $date_to_excl];
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    [$date_from, $date_to, $date_to_excl] = resolve_range_or_fail();

    // required: l2_id
    $l2_id = get_param('l2_id');
    if ($l2_id === null || $l2_id === '') {
        fail('l2_id is required', 400);
    }
    $l2_id = (int)$l2_id;

    // optional filters
    $brand_id = get_param('brand_id');
    $l1_id    = get_param('l1_id'); // اگر خواستی محدود به یک L1 هم باشه

    $city = trim((string)get_param('city', ''));
    $area = trim((string)get_param('area', ''));

    // scope
    $province_scope = !empty($scope['province']) ? (string)$scope['province'] : '';
    $brand_scope    = !empty($scope['brand_id']) ? (int)$scope['brand_id'] : null;
    if ($brand_scope !== null) $brand_id = $brand_scope;

    // area فقط تهران
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    $where  = "WHERE sd.FactorDate >= ? AND sd.FactorDate < ? AND sd.ProductCategoryID = ?";
    $params = [$date_from, $date_to_excl, $l2_id];

    if ($brand_id !== null && $brand_id !== '') { $where .= " AND sd.BrandID = ?"; $params[] = (int)$brand_id; }
    if ($l1_id !== null && $l1_id !== '')       { $where .= " AND sd.ProductGroupID = ?"; $params[] = (int)$l1_id; }

    if ($province_scope !== '') { $where .= " AND sd.Province = ?"; $params[] = $province_scope; }
    if ($city !== '')           { $where .= " AND sd.City = ?";     $params[] = $city; }
    if ($area !== '')           { $where .= " AND sd.Area = ?";     $params[] = $area; }

    $sql = "
        SELECT TOP (5)
            sd.ProductID AS product_id,
            MAX(sd.ProductTitle) AS product_title,
            sd.BrandID AS brand_id,
            MAX(sd.BrandTitle) AS brand_title,
            CAST(SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) AS varchar(60)) AS sales,
            CAST(SUM(TRY_CONVERT(decimal(38,6), sd.Quantity)) AS varchar(60)) AS qty
        FROM BI.SaleDetail sd WITH (NOLOCK)
        $where
          AND sd.ProductID IS NOT NULL
        GROUP BY sd.ProductID, sd.BrandID
        ORDER BY SUM(TRY_CONVERT(decimal(38,0), sd.TotalPrice)) DESC
        OPTION (RECOMPILE);
    ";

    $stmt = sqlsrv_query($conn, $sql, $params);
    if ($stmt === false) {
        fail('DB error (top products l2)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sql]);
    }

    $items = [];
    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $items[] = [
            'product_id' => (int)$r['product_id'],
            'product_title' => (string)($r['product_title'] ?? ''),
            'brand_id' => (int)($r['brand_id'] ?? 0),
            'brand_title' => (string)($r['brand_title'] ?? ''),
            'sales' => (string)($r['sales'] ?? '0'),
            'qty' => (string)($r['qty'] ?? '0'),
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'l2_id' => $l2_id,
            'items' => $items
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $province_scope ?: null,
                'brand_id' => $brand_scope,
            ],
            'filters' => [
                'month_from' => get_param('month_from') ?: null,
                'month_to'   => get_param('month_to') ?: null,
                'date_from'  => $date_from,
                'date_to'    => $date_to,
                'l2_id'      => $l2_id,
                'brand_id'   => ($brand_id !== null && $brand_id !== '') ? (int)$brand_id : null,
                'l1_id'      => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'city'       => $city ?: null,
                'area'       => $area ?: null,
                'top'        => 5,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.SaleDetail (L2=ProductCategoryID)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
