<?php
// /api/brands/rank.php  (Chart2: Rank window = 2 above + selected + 2 below)
define('BI_API', true);

require_once __DIR__ . '/../../src/config/db.php';
require_once __DIR__ . '/../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../src/helpers/Auth.php';

function add_days(string $d, int $days): string {
    return date('Y-m-d', strtotime($d . " {$days} day"));
}

function resolve_range_from_month_or_date($conn): array {
    $month_from = trim((string)get_param('month_from', ''));
    $month_to   = trim((string)get_param('month_to', ''));

    $date_from = parse_date(get_param('date_from'));
    $date_to   = parse_date(get_param('date_to'));

    if ($month_from !== '' && $month_to !== '') {
        if (!preg_match('/^\d{4}\-\d{2}$/', $month_from) || !preg_match('/^\d{4}\-\d{2}$/', $month_to)) {
            fail('Invalid month_from/month_to. Use YYYY-MM', 400);
        }
        $date_from = $month_from . '-01';
        $date_to   = date('Y-m-t', strtotime($month_to . '-01'));
        $date_to_excl = add_days($date_to, 1);
        return [$date_from, $date_to, $date_to_excl, 'month'];
    }

    if ($date_from && $date_to) {
        $date_to_excl = add_days($date_to, 1);
        return [$date_from, $date_to, $date_to_excl, 'date'];
    }

    // fallback: last 90 days by max DateKey
    $stmtMax = sqlsrv_query($conn, "SELECT MAX(DateKey) AS mx FROM BI.AggDailyStoreBrandL1");
    if ($stmtMax === false) fail('DB error (max date)', 500, ['sqlsrv_errors' => sqlsrv_errors()]);
    $mx = sqlsrv_fetch_array($stmtMax, SQLSRV_FETCH_ASSOC)['mx'] ?? null;
    if (!$mx) fail('No data to determine date range', 400);

    $date_to = (string)$mx;
    $date_from = add_days($date_to, -89);
    $date_to_excl = add_days($date_to, 1);

    return [$date_from, $date_to, $date_to_excl, 'fallback_90d'];
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);
    $conn  = db_connect();

    [$date_from, $date_to, $date_to_excl, $range_mode] = resolve_range_from_month_or_date($conn);

    // filters
    $l1_id        = get_param('l1_id');
    $brand_id     = get_param('brand_id'); // selected
    $sale_type_id = get_param('sale_type_id');

    $province = trim((string)get_param('province', ''));
    $city     = trim((string)get_param('city', ''));
    $area     = trim((string)get_param('area', ''));

    // scope
    $province_scope = !empty($scope['province']) ? (string)$scope['province'] : '';
    $brand_scope    = !empty($scope['brand_id']) ? (int)$scope['brand_id'] : null;

    if ($province_scope !== '') $province = $province_scope;
    if ($brand_scope !== null)  $brand_id = $brand_scope;

    // area only for Tehran
    if ($city === '' || ($city !== 'تهران' && strtolower($city) !== 'tehran')) {
        $area = '';
    }

    $brand_id_int = null;
    if ($brand_id !== null && $brand_id !== '') $brand_id_int = (int)$brand_id;

    // WHERE/JOIN
    $joinStore = "";
    $joinBrand = "LEFT JOIN BI.DimBrand db ON db.BrandID = a.BrandID";

    $where  = "WHERE a.DateKey >= ? AND a.DateKey < ?";
    $params = [$date_from, $date_to_excl];

    if ($l1_id !== null && $l1_id !== '') {
        $where .= " AND a.L1ID = ?";
        $params[] = (int)$l1_id;
    }
    if ($sale_type_id !== null && $sale_type_id !== '') {
        $where .= " AND a.SaleTypeID = ?";
        $params[] = (int)$sale_type_id;
    }

    if ($province !== '' || $city !== '' || $area !== '') {
        $joinStore = "JOIN BI.DimStore ds ON ds.StoreID = a.StoreID";
        if ($province !== '') { $where .= " AND ds.Province = ?"; $params[] = $province; }
        if ($city !== '')     { $where .= " AND ds.City = ?";     $params[] = $city; }
        if ($area !== '')     { $where .= " AND ds.Area = ?";     $params[] = $area; }
    }

    if ($brand_id_int === null) {
        // Default: Top 6
        $sql = "
            WITH ranked AS (
                SELECT
                    a.BrandID AS brand_id,
                    ISNULL(NULLIF(LTRIM(RTRIM(db.BrandTitle)),''), CONCAT(N'Brand #', a.BrandID)) AS brand_title,
                    SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales,
                    ROW_NUMBER() OVER (ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC) AS rn
                FROM BI.AggDailyStoreBrandL1 a
                $joinStore
                $joinBrand
                $where
                GROUP BY a.BrandID, db.BrandTitle
            ),
            totals AS (SELECT SUM(sales) AS total_sales FROM ranked)
            SELECT TOP (6)
                r.brand_id,
                r.brand_title,
                CAST(r.sales AS varchar(60)) AS sales,
                CAST((CAST(r.sales AS decimal(38,6))*100) / NULLIF(CAST(t.total_sales AS decimal(38,6)),0) AS decimal(9,2)) AS share_percent,
                r.rn AS rank,
                CAST(0 AS bit) AS is_selected,
                CAST(t.total_sales AS varchar(60)) AS total_sales
            FROM ranked r
            CROSS JOIN totals t
            ORDER BY r.sales DESC
            OPTION (RECOMPILE);
        ";
        $stmt = sqlsrv_query($conn, $sql, $params);
        if ($stmt === false) fail('DB error (rank)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sql]);

    } else {
        // ✅ Rank window ONLY (5 rows around selected)
        $sql = "
            WITH ranked AS (
                SELECT
                    a.BrandID AS brand_id,
                    ISNULL(NULLIF(LTRIM(RTRIM(db.BrandTitle)),''), CONCAT(N'Brand #', a.BrandID)) AS brand_title,
                    SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) AS sales,
                    ROW_NUMBER() OVER (ORDER BY SUM(TRY_CONVERT(decimal(38,0), a.SalesAmount)) DESC) AS rn
                FROM BI.AggDailyStoreBrandL1 a
                $joinStore
                $joinBrand
                $where
                GROUP BY a.BrandID, db.BrandTitle
            ),
            totals AS (
                SELECT COUNT(1) AS total_rows, SUM(sales) AS total_sales FROM ranked
            ),
            sel AS (
                SELECT rn AS sel_rn FROM ranked WHERE brand_id = ?
            ),
            b1 AS (
                SELECT
                    t.total_rows,
                    t.total_sales,
                    s.sel_rn,
                    CASE WHEN s.sel_rn IS NULL THEN 1 ELSE CASE WHEN s.sel_rn-2 < 1 THEN 1 ELSE s.sel_rn-2 END END AS start_rn0
                FROM totals t
                CROSS JOIN sel s
            ),
            b2 AS (
                SELECT
                    total_rows,
                    total_sales,
                    sel_rn,
                    start_rn0,
                    CASE
                        WHEN (start_rn0 + 4) > total_rows THEN total_rows
                        ELSE (start_rn0 + 4)
                    END AS end_rn0
                FROM b1
            ),
            bounds AS (
                SELECT
                    total_rows,
                    total_sales,
                    sel_rn,
                    CASE
                        WHEN (end_rn0 - 4) < 1 THEN 1
                        ELSE (end_rn0 - 4)
                    END AS start_rn,
                    end_rn0 AS end_rn
                FROM b2
            )
            SELECT
                r.brand_id,
                r.brand_title,
                CAST(r.sales AS varchar(60)) AS sales,
                CAST((CAST(r.sales AS decimal(38,6))*100) / NULLIF(CAST(b.total_sales AS decimal(38,6)),0) AS decimal(9,2)) AS share_percent,
                r.rn AS rank,
                CAST(CASE WHEN r.brand_id = ? THEN 1 ELSE 0 END AS bit) AS is_selected,
                CAST(b.total_sales AS varchar(60)) AS total_sales
            FROM ranked r
            CROSS JOIN bounds b
            WHERE r.rn BETWEEN b.start_rn AND b.end_rn
            ORDER BY r.rn ASC
            OPTION (RECOMPILE);
        ";

        $paramsExec = array_merge($params, [$brand_id_int, $brand_id_int]);
        $stmt = sqlsrv_query($conn, $sql, $paramsExec);
        if ($stmt === false) fail('DB error (rank)', 500, ['sqlsrv_errors' => sqlsrv_errors(), 'sql' => $sql]);
    }

    $items = [];
    $selected = null;
    $total_sales = "0";

    while ($r = sqlsrv_fetch_array($stmt, SQLSRV_FETCH_ASSOC)) {
        $total_sales = (string)($r['total_sales'] ?? '0');
        $row = [
            'brand_id' => (int)$r['brand_id'],
            'brand_title' => (string)$r['brand_title'],
            'sales' => (string)($r['sales'] ?? '0'),
            'share_percent' => (float)($r['share_percent'] ?? 0),
            'rank' => (int)($r['rank'] ?? 0),
            'is_selected' => (bool)($r['is_selected'] ?? false),
        ];
        if ($row['is_selected']) $selected = $row;
        $items[] = $row;
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'items' => $items,
            'selected' => $selected,
            'total_sales' => $total_sales,
        ],
        'meta' => [
            'ok' => true,
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $province_scope ?: null,
                'brand_id' => $brand_scope,
            ],
            'filters' => [
                'month_from' => get_param('month_from') ?: null,
                'month_to'   => get_param('month_to') ?: null,
                'date_from'  => $date_from,
                'date_to'    => $date_to,
                'range_mode' => $range_mode,

                'l1_id' => ($l1_id !== null && $l1_id !== '') ? (int)$l1_id : null,
                'brand_id' => $brand_id_int,
                'province' => ($province !== '' ? $province : null),
                'city' => ($city !== '' ? $city : null),
                'area' => ($area !== '' ? $area : null),
                'sale_type_id' => ($sale_type_id !== null && $sale_type_id !== '') ? (int)$sale_type_id : null,
            ],
            'duration_ms' => $elapsed_ms,
            'amount_unit' => 'IRR',
            'source' => 'BI.AggDailyStoreBrandL1 (+DimStore/+DimBrand)',
        ],
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
