<?php
// /api/ai/insight/dashboard.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';
require_once __DIR__ . '/../../../src/helpers/GapGPT.php';


function n($v) {
    if ($v === null) return null;
    if (is_numeric($v)) return (float)$v;
    if (is_string($v)) {
        $x = preg_replace('/[^\d\.\-]/', '', $v);
        return is_numeric($x) ? (float)$x : null;
    }
    return null;
}
function fmt_money($v) {
    $x = n($v);
    if ($x === null) return null;
    return number_format($x, 0) . ' ریال';
}
function top_item($arr) {
    if (!is_array($arr)) return null;
    if (isset($arr['items']) && is_array($arr['items']) && count($arr['items']) > 0) return $arr['items'][0];
    if (count($arr) > 0) return $arr[0];
    return null;
}
function safe_items($arr) {
    if (!is_array($arr)) return [];
    if (isset($arr['items']) && is_array($arr['items'])) return $arr['items'];
    return $arr;
}

function call_context(array $params) : array {
    $qs = http_build_query(array_filter($params, fn($v) => $v !== null && $v !== ''));
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $url = "$scheme://$host/dashboard-project/api/ai/context/dashboard.php" . ($qs ? "?$qs" : "");

    $cookie = $_SERVER['HTTP_COOKIE'] ?? '';

    $opts = [
        'http' => [
            'method' => 'GET',
            'timeout' => 30,
            'ignore_errors' => true,
            'header' => "Accept: application/json\r\n" . ($cookie ? "Cookie: $cookie\r\n" : "")
        ]
    ];
    $ctx = stream_context_create($opts);
    $raw = @file_get_contents($url, false, $ctx);

    if ($raw === false) return ['data' => null, 'meta' => ['ok' => false, 'error' => 'Failed calling context', 'url' => $url]];
    $arr = json_decode($raw, true);
    if (!is_array($arr)) return ['data' => null, 'meta' => ['ok' => false, 'error' => 'Invalid JSON from context', 'url' => $url, 'raw' => $raw]];

    return $arr;
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    $date_from = get_param('date_from');
    $date_to   = get_param('date_to');
    if (!$date_from || !$date_to) fail('date_from and date_to are required', 400);

    $params = [
        'date_from' => $date_from,
        'date_to'   => $date_to,
        'brand_id'  => get_param('brand_id'),
        'store_id'  => get_param('store_id'),
        'city'      => get_param('city'),
        'area'      => get_param('area'),
        'l1_id'     => get_param('l1_id'),
        'sale_type_id' => get_param('sale_type_id'),
        'province'  => get_param('province'),
    ];

    // scope overrides
    if (!empty($scope['brand_id'])) $params['brand_id'] = (int)$scope['brand_id'];
    if (!empty($scope['province'])) $params['province'] = (string)$scope['province'];

    $ctx = call_context($params);
    if (empty($ctx['meta']['ok'])) {
        fail('Failed to build context', 500, ['context_meta' => $ctx['meta'] ?? null]);
    }

    $data = $ctx['data'] ?? [];
    $kpi  = $data['kpi'] ?? [];
    $ov   = $data['overview'] ?? [];

    $cards = [];
    $charts_used = [];
    $kpis_used = ['total_sales','last_week_sales','avg_sales_per_store'];

    // Summary
    $total_sales = $kpi['total_sales'] ?? null;
    $last_week_sales = $kpi['last_week_sales'] ?? null;
    $avg_sales_per_store = $kpi['avg_sales_per_store'] ?? null;

    $summary = "بازه {$date_from} تا {$date_to}";
    if ($total_sales !== null) $summary .= " | فروش کل: " . fmt_money($total_sales);
    if ($last_week_sales !== null) $summary .= " | فروش هفته گذشته: " . fmt_money($last_week_sales);
    if ($avg_sales_per_store !== null) $summary .= " | میانگین هر فروشگاه: " . fmt_money($avg_sales_per_store);

    // Card: Top brand
    if (!empty($ov['top_brands_month'])) {
        $charts_used[] = 'top_brands_month';
        $topBrand = top_item($ov['top_brands_month']);
        if ($topBrand) {
            $bn = $topBrand['brand_title'] ?? '—';
            $bs = fmt_money($topBrand['sales'] ?? null) ?? '—';
            $bp = $topBrand['share_percent'] ?? null;
            $txt = "برند «{$bn}» با فروش {$bs}" . ($bp !== null ? " و سهم {$bp}% در صدر قرار دارد." : " در صدر قرار دارد.");
            $cards[] = [
                'type' => 'insight',
                'severity' => 'info',
                'title' => 'برند برتر',
                'text' => $txt,
                'refs' => ['top_brands_month'],
            ];

            if ($bp !== null && (float)$bp >= 20) {
                $cards[] = [
                    'type' => 'alert',
                    'severity' => 'warning',
                    'title' => 'تمرکز فروش روی یک برند',
                    'text' => "سهم برند «{$bn}» بالاتر از ۲۰٪ است؛ در صورت افت تأمین/تقاضا ریسک نوسان بالا می‌رود.",
                    'refs' => ['top_brands_month','brand_share_14d'],
                ];
            }
        }
    }

    // Card: Top L1
    if (!empty($ov['l1_share'])) {
        $charts_used[] = 'l1_share';
        $topL1 = top_item($ov['l1_share']);
        if ($topL1) {
            $gn = $topL1['l1_title'] ?? '—';
            $gs = fmt_money($topL1['sales'] ?? null) ?? '—';
            $gp = $topL1['share_percent'] ?? null;
            $cards[] = [
                'type' => 'insight',
                'severity' => 'info',
                'title' => 'گروه کالایی برتر (L1)',
                'text' => "گروه «{$gn}» با فروش {$gs}" . ($gp !== null ? " و سهم {$gp}% بیشترین سهم را دارد." : " بیشترین سهم را دارد."),
                'refs' => ['l1_share'],
            ];

            $cards[] = [
                'type' => 'recommendation',
                'severity' => 'success',
                'title' => 'اقدام پیشنهادی',
                'text' => "برای افزایش فروش، موجودی و پروموشن گروه «{$gn}» را در فروشگاه‌های پرفروش تقویت کنید و اثر را هفتگی پایش کنید.",
                'refs' => ['l1_share','top_stores'],
            ];
        }
    }

    // Card: brand_share_14d (اگر داشتیم)
    if (!empty($ov['brand_share_14d'])) {
        $charts_used[] = 'brand_share_14d';
        $items = safe_items($ov['brand_share_14d']);
        if (count($items) >= 2) {
            $cards[] = [
                'type' => 'insight',
                'severity' => 'info',
                'title' => 'روند سهم برندها (۱۴ روز اخیر)',
                'text' => "روند سهم فروش برندها در ۱۴ روز اخیر آماده است و می‌توان نوسانات سهم برندهای برتر را بررسی کرد.",
                'refs' => ['brand_share_14d'],
            ];
        }
    } else {
        // تا وقتی endpointش نیومده
        $cards[] = [
            'type' => 'note',
            'severity' => 'info',
            'title' => 'داده ناقص: نمودار ۱۴ روز برندها',
            'text' => "خروجی «brand_share_14d» هنوز در context موجود نیست یا endpoint آن فعال نشده.",
            'refs' => [],
        ];
    }

    // Card: top_stores (اگر داشتیم)
    if (!empty($ov['top_stores'])) {
        $charts_used[] = 'top_stores';
        $items = safe_items($ov['top_stores']);
        if (!empty($items)) {
            $s0 = $items[0];
            $sn = $s0['store_name'] ?? $s0['YaranName'] ?? '—';
            $cards[] = [
                'type' => 'insight',
                'severity' => 'info',
                'title' => 'فروشگاه برتر',
                'text' => "فروشگاه «{$sn}» در صدر فروشگاه‌های منتخب قرار دارد. می‌توان الگوی فروش/سبد کالا را از آن استخراج کرد.",
                'refs' => ['top_stores'],
            ];
        }
    } else {
        $cards[] = [
            'type' => 'note',
            'severity' => 'info',
            'title' => 'داده ناقص: فروشگاه‌های برتر',
            'text' => "خروجی «top_stores» هنوز در context موجود نیست یا endpoint آن فعال نشده.",
            'refs' => [],
        ];
    }

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'summary' => $summary,
            'cards' => $cards,
            'kpis_used' => $kpis_used,
            'charts_used' => array_values(array_unique($charts_used)),
        ],
        'meta' => [
            'ok' => true,
            'engine' => 'ui-cards (rule-based)',
            'role' => $user['Role'] ?? null,
            'scope' => [
                'province' => $scope['province'] ?? null,
                'brand_id' => $scope['brand_id'] ?? null,
            ],
            'filters' => $params,
            'generated_at' => date('Y-m-d H:i:s'),
            'duration_ms' => $elapsed_ms,
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
