<?php
// /api/ai/context/dashboard.php
define('BI_API', true);

require_once __DIR__ . '/../../../src/helpers/helperFunctions.php';
require_once __DIR__ . '/../../../src/helpers/Auth.php';

function call_api(string $path, array $params = []) : array {
    $qs = http_build_query(array_filter($params, fn($v) => $v !== null && $v !== ''));
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST']; // e.g. localhost:81
    $url = "$scheme://$host/dashboard-project/api/$path" . ($qs ? "?$qs" : "");

    $cookie = $_SERVER['HTTP_COOKIE'] ?? '';

    $opts = [
        'http' => [
            'method'  => 'GET',
            'timeout' => 30,
            'header'  => "Accept: application/json\r\n"
                       . ($cookie ? "Cookie: $cookie\r\n" : "")
        ]
    ];
    $ctx = stream_context_create($opts);
    $raw = @file_get_contents($url, false, $ctx);

    // extract http code
    $http_code = null;
    if (!empty($http_response_header) && is_array($http_response_header)) {
        foreach ($http_response_header as $h) {
            if (preg_match('#HTTP/\S+\s+(\d{3})#', $h, $m)) { $http_code = (int)$m[1]; break; }
        }
    }

    if ($raw === false) {
        return ['data' => null, 'meta' => ['ok' => false, 'error' => 'Request failed', 'url' => $url, 'http_code' => $http_code]];
    }

    $arr = json_decode($raw, true);
    if (!is_array($arr)) {
        return ['data' => null, 'meta' => ['ok' => false, 'error' => 'Invalid JSON from endpoint', 'url' => $url, 'http_code' => $http_code, 'raw' => $raw]];
    }

    // inject called_url/http_code if endpoint didn't provide
    if (!isset($arr['meta'])) $arr['meta'] = [];
    if (!isset($arr['meta']['called_url'])) $arr['meta']['called_url'] = $url;
    if (!isset($arr['meta']['http_code']))  $arr['meta']['http_code']  = $http_code;

    return $arr;
}

try {
    $t0 = microtime(true);

    $user  = auth_user();
    $scope = require_role_scope_filters($user);

    // required dates
    $date_from = get_param('date_from');
    $date_to   = get_param('date_to');
    if (!$date_from || !$date_to) fail('date_from and date_to are required', 400);

    // filters
    $brand_id = get_param('brand_id');
    $store_id = get_param('store_id');
    $city     = get_param('city');
    $area     = get_param('area');
    $l1_id    = get_param('l1_id');
    $sale_type_id = get_param('sale_type_id');

    // scope overrides
    $province = get_param('province');
    if (!empty($scope['brand_id']))  $brand_id = (int)$scope['brand_id'];
    if (!empty($scope['province']))  $province = (string)$scope['province'];

    $baseParams = [
        'date_from' => $date_from,
        'date_to'   => $date_to,
        'brand_id'  => $brand_id,
        'store_id'  => $store_id,
        'city'      => $city,
        'area'      => $area,
        'l1_id'     => $l1_id,
        'sale_type_id' => $sale_type_id,
        'province'  => $province,
    ];

    // ✅ use REAL endpoints that exist in your project
    $kpi      = call_api('kpi/summary.php', $baseParams);
    $topBrands= call_api('brands/top3m.php', $baseParams);          // top brands
    $l1Share  = call_api('l1/top5.php', array_merge($baseParams, ['limit' => 5])); // top5 L1 (share)
    $brand14d = call_api('brands/daily_share.php', $baseParams);     // 14d share compare
    $topStores= call_api('stores/top20.php', array_merge($baseParams, ['limit' => 20])); // top stores

    $elapsed_ms = (int)round((microtime(true) - $t0) * 1000);

    json_response([
        'data' => [
            'user' => [
                'role' => $user['Role'] ?? null,
                'user_id' => $user['UserID'] ?? null,
                'username' => $user['Username'] ?? null,
            ],
            'scope' => [
                'province' => $scope['province'] ?? null,
                'brand_id' => $scope['brand_id'] ?? null,
            ],
            'filters' => [
                'date_from' => $date_from,
                'date_to' => $date_to,
                'brand_id' => $brand_id ?: null,
                'store_id' => $store_id ?: null,
                'city' => $city ?: null,
                'area' => $area ?: null,
                'l1_id' => $l1_id ?: null,
                'sale_type_id' => $sale_type_id ?: null,
                'province' => $province ?: null,
            ],
            'kpi' => $kpi['data'] ?? null,
            'overview' => [
                'top_brands_month' => $topBrands['data'] ?? null,
                'l1_share'         => $l1Share['data'] ?? null,
                'brand_share_14d'  => $brand14d['data'] ?? null,
                'top_stores'       => $topStores['data'] ?? null,
            ],
            'sources_meta' => [
                'kpi'      => $kpi['meta'] ?? null,
                'topBrands'=> $topBrands['meta'] ?? null,
                'l1Share'  => $l1Share['meta'] ?? null,
                'brand14d' => $brand14d['meta'] ?? null,
                'topStores'=> $topStores['meta'] ?? null,
            ],
        ],
        'meta' => [
            'ok' => true,
            'generated_at' => date('Y-m-d H:i:s'),
            'duration_ms' => $elapsed_ms,
        ]
    ]);

} catch (Throwable $e) {
    fail('Server error: ' . $e->getMessage(), 500);
}
